//--------------------------------------------------------------------------
//
//  Software for MSP430 based e-meters.
//
//  You may not use the Program in non-TI devices.
//
//  File: emeter-communications.c
//
//  Steve Underwood <steve-underwood@ti.com>
//  Texas Instruments Hong Kong Ltd.
//
//  $Id: emeter-communication.c,v 1.7 2005/11/17 09:25:23 a0754793 Exp $
//
/*! \file emeter-structs.h */
//
//--------------------------------------------------------------------------
//
#include <stdint.h>
#if !defined(__MSP430__)
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#endif
#if defined(__GNUC__)
#include <signal.h>
#endif
#if defined(__MSP430__)
#include <io.h>
#endif
#include <emeter-toolkit.h>
#include "emeter-structs.h"

#if !defined(NULL)
#define NULL    (void *) 0
#endif

#define FSEG_A 0x01080      // Flash Segment A start address
#define FSEG_B 0x01000      // Flash Segment B start address

enum host_commands_e
{
    HOST_CMD_GET_METER_CONFIGURATION            = 0x56,
    HOST_CMD_SET_METER_CONSUMPTION              = 0x57,
    HOST_CMD_SET_RTC                            = 0x58,
    HOST_CMD_GET_RTC                            = 0x59,
    HOST_CMD_SET_PASSWORD                       = 0x60,
    HOST_CMD_GET_READINGS_PHASE_1               = 0x61,
    HOST_CMD_GET_READINGS_PHASE_2               = 0x62,
    HOST_CMD_GET_READINGS_PHASE_3               = 0x63,
    HOST_CMD_GET_READINGS_NEUTRAL               = 0x64,
    HOST_CMD_ERASE_FLASH_SEGMENT                = 0x70,
    HOST_CMD_SET_FLASH_POINTER                  = 0x71,
    HOST_CMD_FLASH_DOWNLOAD                     = 0x72,
    HOST_CMD_FLASH_UPLOAD                       = 0x73,
    HOST_CMD_ZAP_MEMORY_AREA                    = 0x74,
    HOST_CMD_SUMCHECK_MEMORY                    = 0x75,
    HOST_CMD_GET_RAW_POWER_PHASE_1              = 0x91,
    HOST_CMD_GET_RAW_POWER_PHASE_2              = 0x92,
    HOST_CMD_GET_RAW_POWER_PHASE_3              = 0x93,
    HOST_CMD_GET_RAW_REACTIVE_POWER_PHASE_1     = 0x95,
    HOST_CMD_GET_RAW_REACTIVE_POWER_PHASE_2     = 0x96,
    HOST_CMD_GET_RAW_REACTIVE_POWER_PHASE_3     = 0x97,
    HOST_CMD_GET_RAW_POWER_NEUTRAL              = 0x94,//0x99,  Percy
    HOST_CMD_GET_RAW_REACTIVE_POWER_NEUTRAL     = 0x98,//0x9D,
    HOST_CMD_CHECK_RTC_ERROR                    = 0xA0,
    HOST_CMD_RTC_CORRECTION                     = 0xA1,
    HOST_CMD_MULTIRATE_SET_PARAMETERS           = 0xC0,
    HOST_CMD_MULTIRATE_GET_PARAMETERS           = 0xC1,
    HOST_CMD_MULTIRATE_CLEAR_USAGE              = 0xC2,
    HOST_CMD_MULTIRATE_GET_USAGE                = 0xC3,
    HOST_CMD_CALIBRATION_GET_GAIN              = 0xE0,         //Percy for Calibration 0902
    HOST_CMD_CALIBRATION_SET_POFFSET             = 0xE1,
    HOST_CMD_CALIBRATION_SET_GAIN               = 0xE2,         
    HOST_CMD_CALIBRATION_SET_PHASE              = 0xE3,
    HOST_CMD_CALIBRATION_STORE                  = 0xE4,
   
};

#if defined(__MSP430__)  &&  (defined(IEC1107_SUPPORT)  ||  defined(SERIAL_CALIBRATION_SUPPORT))
#define MAX_IEC1107_MSG_BODY        32
typedef union
{
#if 0
    unsigned char uint8[4 + 12 + 256];
    unsigned int uint16[(4 + 12 + 256)/2];
#else
    unsigned char uint8[4 + 12 + MAX_IEC1107_MSG_BODY];
    unsigned int uint16[(4 + 12 + MAX_IEC1107_MSG_BODY)/2];
#endif
} serial_msg_t;

#define IEC1107_MSG_RX_START_BODY   10
#define IEC1107_MSG_RX_START_BODY_W 5
#define IEC1107_MSG_TX_START_BODY   14
#define IEC1107_MSG_TX_START_BODY_W 7

/* Incoming serial message buffer */
serial_msg_t rx_msg;
unsigned char rx_msg_len;
unsigned char char_timeout_1107;

static unsigned char rx_msg_ptr;

/* Outgoing serial message buffer */
serial_msg_t tx_msg;
unsigned char tx_msg_len;

static unsigned char tx_msg_ptr;

unsigned int *next_flash_loc;

#endif

int is_calibration_enabled(void)
{
    return TRUE;
}



//////////////////////////Percy Yu 0902///////////////////////////////////////
//Function: Store the calibrated value to flash
//Parameter:  
//bi_work     1: 32bit calibration data
//            0: 16bit calibration data
//value       calibration data to be writen
//offset      Offset to 0x1000, with below format
//Data Format: Start from 0x1000          Offset(in 16bit int)
//    Live    initial_I_dc_estimate       0
//            Ac_offset                   2
//            I_rms_scale_factor          4
//            P_scale_factor              5
//            Phase_correction            6
//            Offset_active_power         7
//            Offset_reactive_power       8
//    Neutral initial_I_dc_estimate       0+9
//            Ac_offset                   2+9
//            I_rms_scale_factor          4+9
//            P_scale_factor              5+9
//            Phase_correction            6+9
//            Offset_active_power         7+9
//            Offset_reactive_power       8+9
//            V_rms_scale_factor          9+9
//            initial_v_dc_estimate       10+9
//////////////////////////////////////////////////////////////////////////////
static void DataStore(unsigned char offset, int32_t value, unsigned char bi_word)
{
    int *read_ptr;
    int *write_ptr;
    int w;

    flash_clr((int *) FSEG_A);

    _DINT();
    //Set to write mode to prepare for copy
    FCTL3 = FWKEY;          /* Lock = 0 */
    FCTL1 = FWKEY | WRT;

    //Copy block B to A for back up
    read_ptr = (int *) FSEG_B;
    write_ptr = (int *) FSEG_A;
    for (w = 0;  w < 64;  w++)
        *write_ptr++ = *read_ptr++;
    
    flash_clr((int *) FSEG_B);

    //Set to write mode to prepare for copy
    FCTL3 = FWKEY;          /* Lock = 0 */
    FCTL1 = FWKEY | WRT;
    
    write_ptr = (int *) FSEG_B + offset;
    if(bi_word == 0)
      *write_ptr = (int)value&0xffff;
    else{
      *write_ptr++ = (int)value&0xffff;
      *write_ptr = (int)(value>>16);
    }
    
    //Copy block A to B, slipping in the new value at the right location
    read_ptr = (int *) FSEG_A;
    write_ptr = (int *) FSEG_B;
    for (w = 0;  w < 64;  w++, read_ptr++, write_ptr++)
    {
        if (*write_ptr == 0xffff)
            *write_ptr = *read_ptr;
    }
    flash_secure();
    _EINT();

}
//////////////////////////Percy Yu 0902///////////////////////////////////////
//Function: calibrate gain scale factor for I channel
//int error: the error from host, in terms of 0.01%
//            For example if real error is 1.5%, error = 150
//unsigned char ch: channel number  0,1,2
//unsigned char stage:  amplifier stage 1,2
//////////////////////////////////////////////////////////////////////////////
void Cal_Gain(short error, unsigned char ch, unsigned char stage){
  
        float tmp;      
        int value;
        struct current_sensor_nv_parms_s * phase_info;  
        unsigned char offset = P_SCALE_FACTOR_OFFSET;
/*        
        if((signed char)error<0)          
          tmp = 1-(256-error)/(float)1000;
        else
          tmp = 1+error/(float)1000;
*/        
     
        if( ch == 3 )   //neutral 
        {
          phase_info =(struct current_sensor_nv_parms_s *) &nv_parms.seg_a.s.chan1.current[1]; 
          offset += CURRENT_SENSOR_NV_PARMS_S_SIZE_W; 
        }
        else{
          phase_info =(struct current_sensor_nv_parms_s *) &nv_parms.seg_a.s.chan1.current[0];  
        }
        value = phase_info->P_scale_factor[0];
        tmp = (value/(1+(signed short)error/(float)10000));
        value = (unsigned int)tmp;
        if(tmp - value > 0.5)
          value += 1;     
        
        DataStore(offset, value, 0);

}

//////////////////////////Percy Yu 0902///////////////////////////////////////
//Function: calibrate phase shift for I channel
//char phase_error: the error from host, in terms of 1 step
//unsigned char ch: channel number  0,1,2
//unsigned char stage:  amplifier stage 1,2
//////////////////////////////////////////////////////////////////////////////
void Cal_Phase(short phase_error, unsigned char ch, unsigned char stage){
  
        int value;
        struct current_sensor_nv_parms_s * phase_info;  
        unsigned char offset = PHASE_CORRECTION_OFFSET;  
        
        #if defined(NEUTRAL_MONITOR_SUPPORT)
        if( ch == 3 )   //neutral 
        {
          offset += CURRENT_SENSOR_NV_PARMS_S_SIZE_W; 

          phase_info =(struct current_sensor_nv_parms_s *) &nv_parms.seg_a.s.chan1.current[1]; 
          value = phase_info->Phase_correction[0] - (signed short)phase_error;
          phase->neutral.in_phase_correction[0].step = I_HISTORY_STEPS - (value>>8);
          phase->neutral.in_phase_correction[0].sd16_preloaded_offset = value;
          SD16PRE_NEUTRAL = (phase_error) & 0xFF;
        }
        else
        #endif
        {
          phase_info =(struct current_sensor_nv_parms_s *) &nv_parms.seg_a.s.chan1.current[0];  
          value = phase_info->Phase_correction[0] - (signed short)phase_error;
          phase->current.in_phase_correction[0].step = I_HISTORY_STEPS - (value>>8);
          phase->current.in_phase_correction[0].sd16_preloaded_offset = value;
          SD16PRE_LIVE = (phase_error) & 0xFF;
        }        
        
        DataStore(offset, value, 0);   
        
}

//////////////////////////Percy Yu 0909///////////////////////////////////////
//Function: calibrate power offset for I channel
//int phase_error: the new poffset
//unsigned char ch: channel number  0,1,2,3(neutral for 1p emeter)
//unsigned char stage:  amplifier stage 1,2
//////////////////////////////////////////////////////////////////////////////
void Cal_Poffset(int phase_error, unsigned char ch, unsigned char stage){
        
    unsigned char offset = OFFSET_ACTIVE_POWER_OFFSET;   
           
    #if defined(NEUTRAL_MONITOR_SUPPORT)
    if(ch == 3){
      offset += CURRENT_SENSOR_NV_PARMS_S_SIZE_W;       
    }      
    #endif         
    
    DataStore(offset, phase_error, 0);     

}


#ifdef IEC1107_D_SUPPORT
int iec1107d_status;

void send_1107d_report(void)
{
    int i;
    unsigned char x;
    unsigned char bcd[5];
    const unsigned char *s;
    static const unsigned char id[] = "/SJJ5eHZSJJ V1.01\r\n";
    static const unsigned char serial_number_tag[] = "1-0:0.0.9*255(";
    static const unsigned char property_number_tag[] = "1-0:0.0.0*255(";
    static const unsigned char energy_tag[] = "1-0:1.8.0*255(";
    static const unsigned char status_tag[] = "1-0:96.5.5*255(";
    static const unsigned char factory_number_tag[] = "0-0:96.1.255*255(";
    static const unsigned char even_parity[128] =
    {
        0x00, 0x81, 0x82, 0x03, 0x84, 0x05, 0x06, 0x87,
        0x88, 0x09, 0x0A, 0x8B, 0x0C, 0x8D, 0x8E, 0x0F,
        0x90, 0x11, 0x12, 0x93, 0x14, 0x95, 0x96, 0x17,
        0x18, 0x99, 0x9A, 0x1B, 0x9C, 0x1D, 0x1E, 0x9F,
        0xA0, 0x21, 0x22, 0xA3, 0x24, 0xA5, 0xA6, 0x27,
        0x28, 0xA9, 0xAA, 0x2B, 0xAC, 0x2D, 0x2E, 0xAF,
        0x30, 0xB1, 0xB2, 0x33, 0xB4, 0x35, 0x36, 0xB7,
        0xB8, 0x39, 0x3A, 0xBB, 0x3C, 0xBD, 0xBE, 0x3F,
        0xC0, 0x41, 0x42, 0xC3, 0x44, 0xC5, 0xC6, 0x47,
        0x48, 0xC9, 0xCA, 0x4B, 0xCC, 0x4D, 0x4E, 0xCF,
        0x50, 0xD1, 0xD2, 0x53, 0xD4, 0x55, 0x56, 0xD7,
        0xD8, 0x59, 0x5A, 0xDB, 0x5C, 0xDD, 0xDE, 0x5F,
        0x60, 0xE1, 0xE2, 0x63, 0xE4, 0x65, 0x66, 0xE7,
        0xE8, 0x69, 0x6A, 0xEB, 0x6C, 0xED, 0xEE, 0x6F,
        0xF0, 0x71, 0x72, 0xF3, 0x74, 0xF5, 0xF6, 0x77,
        0x78, 0xF9, 0xFA, 0x7B, 0xFC, 0x7D, 0x7E, 0xFF,
    };

    /* IEC1107 specifies 7 bit + even parity data. We want to use 8 bit data for calibration, so we
       use software generation for the parity bit. */
    /* If the password is set we are in calibration mode, and the IEC1107D messages should not be sent.
       After calibration is complete, the password may or may not have been cleared. If it is still set
       a reset of the MCU will restore normal operation. */
    if ((meter_status & PASSWORD_OK))
        return;
    iec1107d_status = 0;
    if (total_power > 0)
        iec1107d_status |= 0x80;
#if SELF_TEST_SUPPORT
    if (nv_parms.s.meter_failures != 0xFFFF)
        iec1107d_status |= 0x01;
#endif

    i = 0;
    s = id;
    while (*s)
        tx_msg.uint8[i++] = even_parity[*s++];

    tx_msg.uint8[i++] = even_parity['\r'];
    tx_msg.uint8[i++] = even_parity['\n'];

    if (nv_parms.s.property_number[0])
    {
        s = property_number_tag;
        while (*s)
            tx_msg.uint8[i++] = even_parity[*s++];
        s = nv_parms.s.property_number;
        while (*s)
            tx_msg.uint8[i++] = even_parity[*s++];
        tx_msg.uint8[i++] = even_parity[')'];
        tx_msg.uint8[i++] = even_parity['\r'];
        tx_msg.uint8[i++] = even_parity['\n'];
    }

    if (nv_parms.s.serial_number[0])
    {
        s = serial_number_tag;
        while (*s)
            tx_msg.uint8[i++] = even_parity[*s++];
        s = nv_parms.s.serial_number;
        while (*s)
            tx_msg.uint8[i++] = even_parity[*s++];
        tx_msg.uint8[i++] = even_parity[')'];
        tx_msg.uint8[i++] = even_parity['\r'];
        tx_msg.uint8[i++] = even_parity['\n'];
    }

    s = energy_tag;
    while (*s)
        tx_msg.uint8[i++] = even_parity[*s++];

    last_reported_total_consumed_kwh = total_consumed_kwh;
    last_reported_total_consumed_100mwh = total_consumed_100mwh;

    bin2bcd32(bcd, last_reported_total_consumed_kwh);
    tx_msg.uint8[i++] = even_parity['0' + ((bcd[2] >> 4) & 0xF)];
    tx_msg.uint8[i++] = even_parity['0' + (bcd[2] & 0xF)];
    tx_msg.uint8[i++] = even_parity['0' + ((bcd[3] >> 4) & 0xF)];
    tx_msg.uint8[i++] = even_parity['0' + (bcd[3] & 0xF)];
    tx_msg.uint8[i++] = even_parity['0' + ((bcd[4] >> 4) & 0xF)];
    tx_msg.uint8[i++] = even_parity['0' + (bcd[4] & 0xF)];
    tx_msg.uint8[i++] = even_parity['.'];
    bin2bcd16(bcd, last_reported_total_consumed_100mwh);
    tx_msg.uint8[i++] = even_parity['0' + ((bcd[1] >> 4) & 0xF)];
    tx_msg.uint8[i++] = even_parity['0' + (bcd[1] & 0xF)];
    tx_msg.uint8[i++] = even_parity['0' + ((bcd[2] >> 4) & 0xF)];
    tx_msg.uint8[i++] = even_parity['0' + (bcd[2] & 0xF)];
    tx_msg.uint8[i++] = even_parity[')'];
    tx_msg.uint8[i++] = even_parity['\r'];
    tx_msg.uint8[i++] = even_parity['\n'];

    s = status_tag;
    while (*s)
        tx_msg.uint8[i++] = even_parity[*s++];
    x = '0' + ((iec1107d_status >> 4) & 0xF);
    if (x > '9')
        x += 7;
    tx_msg.uint8[i++] = even_parity[x];
    x = '0' + (iec1107d_status & 0xF);
    if (x > '9')
        x += 7;
    tx_msg.uint8[i++] = even_parity[x];
    tx_msg.uint8[i++] = even_parity[')'];
    tx_msg.uint8[i++] = even_parity['\r'];
    tx_msg.uint8[i++] = even_parity['\n'];

    if (nv_parms.s.factory_number[0])
    {
        s = factory_number_tag;
        while (*s)
            tx_msg.uint8[i++] = even_parity[*s++];
        s = nv_parms.s.factory_number;
        while (*s)
            tx_msg.uint8[i++] = even_parity[*s++];
        tx_msg.uint8[i++] = even_parity[')'];
        tx_msg.uint8[i++] = even_parity['\r'];
        tx_msg.uint8[i++] = even_parity['\n'];
    }

    tx_msg.uint8[i++] = even_parity['!'];
    tx_msg.uint8[i++] = even_parity['\r'];
    tx_msg.uint8[i++] = even_parity['\n'];
    tx_msg_ptr = 0;
    tx_msg_len = i;

    #if defined(USART1TX_VECTOR)
    IE2 |= UTXIE1;
    #else
    U0IE |= UTXIE0;
    #endif
}
#endif

#if defined(__MSP430__)  &&  (defined(IEC1107_SUPPORT)  ||  defined(SERIAL_CALIBRATION_SUPPORT))
int send_msg(int len)
{
    if (len > 4 + 12 + MAX_IEC1107_MSG_BODY)
        return FALSE;
    tx_msg_ptr = 0;
    tx_msg_len = len;
    #if defined(USART1TX_VECTOR)
    IE2 |= UTXIE1;
    #else
    U0IE |= UTXIE0;
    #endif
    return  TRUE;
}

int prepare_tx_message(int len)
{
    int i;

    tx_msg.uint8[0] = 0xFE;
    tx_msg.uint8[1] = 0xFE;
    tx_msg.uint8[2] = 0xFE;
    tx_msg.uint8[3] = 0xFE;
    tx_msg.uint8[4] = 0x68;
    tx_msg.uint8[5] = 0x99;
    tx_msg.uint8[6] = 0x99;
    tx_msg.uint8[7] = 0x99;
    tx_msg.uint8[8] = 0x99;
    tx_msg.uint8[9] = 0x99;
    tx_msg.uint8[10] = 0x99;
    tx_msg.uint8[11] = 0x68;
    tx_msg.uint8[12] = 0x23;
    tx_msg.uint8[13] = len;
    tx_msg.uint8[IEC1107_MSG_TX_START_BODY + len] = 0;
    tx_msg.uint8[IEC1107_MSG_TX_START_BODY + len + 1] = 0x16;
    for (i = 4;  i < IEC1107_MSG_TX_START_BODY + len;  i++)
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + len] += tx_msg.uint8[i];
    return send_msg(IEC1107_MSG_TX_START_BODY + len + 2);
}

void process_rx_message(serial_msg_t *rx_msg, int rx_len)
{
    int i;
    long int z;
    long int z1;
    #if !defined(SINGLE_PHASE)
    struct phase_parms_s *phase;
    #endif
    unsigned int *last_flash_loc;

    /* Messages with type 0x23 are custom messages we
      use for calibration, password protection, etc.
      All other message types go to a custom message
      handler (if available). */
    if (rx_msg->uint8[8] != 0x23)
    {
#ifdef CUSTOM_SERIAL_MESSAGE_SUPPORT
        custom_serial_message_handler(&rx_msg, rx_msg_len);
#endif
        return;
    }
    if ((rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] & 0x80))
    {
        /* This looks like one of our own messages, which has echoed back
           to us */
        return;
    }

    /* Only process messages if the password has been given correctly
       (except for the password test message, of course). */
//    if (!(meter_status & PASSWORD_OK)  &&  rx_msg->uint8[IEC1107_MSG_RX_START_BODY] != 0x60)
//        return;

    switch (rx_msg->uint8[IEC1107_MSG_RX_START_BODY])
    {
    case HOST_CMD_GET_METER_CONFIGURATION:
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 2] = NUM_PHASES;
    #if defined(NEUTRAL_MONITOR_SUPPORT)
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 3] = 1;
    #else
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 3] = 0;
    #endif
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 4] = 0

                                                    | 0x02

    #if defined(DYNAMIC_PHASE_CORRECTION_SUPPORT)
                                                    | 0x04
    #endif
    #if defined(RTC_SUPPORT)
                                                    | 0x08
    #endif
    #if defined(CORRECTED_RTC_SUPPORT)
                                                    | 0x10
    #endif
    #if defined(TEMPERATURE_SUPPORT)
                                                    | 0x20
    #endif
    #ifdef SELF_TEST_SUPPORT
                                                    | 0x40
    #endif
    #ifdef MULTI_RATE_SUPPORT
                                                    | 0x80
    #endif
                                                    ;
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 5] = 0x01
    #if REACTIVE_POWER_SUPPORT
                                                    | 0x02
    #endif
    #if defined(VA_POWER_SUPPORT)
                                                    | 0x04
    #endif
    #if defined(VRMS_SUPPORT)
                                                    | 0x08
    #endif
    #if defined(IRMS_SUPPORT)
                                                    | 0x10
    #endif
    #if defined(POWER_FACTOR_SUPPORT)
                                                    | 0x20
    #endif
    #if defined(MAINS_FREQUENCY_SUPPORT)
                                                    | 0x40
    #endif
                                                    ;
        prepare_tx_message(6);
        break;
    case HOST_CMD_SET_METER_CONSUMPTION:
        z = rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 1];
        z |= (long int) rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 2] << 16;
        z1 = rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 3];
        z1 |= (long int) rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 4] << 16;
        //custom_set_consumption(z, z1);
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        prepare_tx_message(2);
    case HOST_CMD_SET_RTC:
    #if defined(RTC_SUPPORT)
        rtc.year = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 2];
        rtc.month = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 3];
        rtc.day = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 4];
        rtc.hour = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 5];
        rtc.minute = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 6];
        rtc.second = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 7];
        set_rtc_sumcheck();
    #endif
    #if defined(CUSTOM_RTC_SUPPORT)
        custom_rtc_set(rx_msg->uint8);
    #endif
    #if defined(MULTI_RATE_SUPPORT)
        multirate_align_with_rtc();
    #endif
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        prepare_tx_message(2);
        break;
    case HOST_CMD_GET_RTC:
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
    #if defined(RTC_SUPPORT)
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 2] = rtc.year;
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 3] = rtc.month;
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 4] = rtc.day;
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 5] = rtc.hour;
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 6] = rtc.minute;
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 7] = rtc.second;
    #endif
    #if defined(CUSTOM_RTC_SUPPORT)
        custom_rtc_retrieve(tx_msg.uint8);
    #endif
    #if defined(TEMPERATURE_SUPPORT)
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 4] = temperature;
    #endif
        prepare_tx_message(10);
        break;
    case HOST_CMD_SET_PASSWORD:
        /* Check the calibration password */
        if (rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 1] == SERIAL_CALIBRATION_PASSWORD_1
            &&
            rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 2] == SERIAL_CALIBRATION_PASSWORD_2
            &&
            rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 3] == SERIAL_CALIBRATION_PASSWORD_3
            &&
            rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 4] == SERIAL_CALIBRATION_PASSWORD_4)
        {
            meter_status |= PASSWORD_OK;
            tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
            tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
            prepare_tx_message(2);
        }
        else
        {
            /* Only respond to a bad password, if the password was good before. That lets
               us know we have unset the password OK, but doesn't give any information to
               people trying to attack the meter. */
            if ((meter_status & PASSWORD_OK))
            {
                tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
                tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
                prepare_tx_message(2);
            }
            meter_status &= ~PASSWORD_OK;
        }
        break;
    case HOST_CMD_GET_READINGS_PHASE_1:
    #if !defined(SINGLE_PHASE)
    case HOST_CMD_GET_READINGS_PHASE_2:
    case HOST_CMD_GET_READINGS_PHASE_3:
        /* Exchange voltage, current and power readings (neutral).
           frequency, power factor and reactive power readings. */
        phase = &chan[rx_msg->uint8[IEC1107_MSG_RX_START_BODY] - 0x61];
    #endif
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
    #if defined(VRMS_SUPPORT)
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = phase->V_rms;
    #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = 0;
    #endif
    #if defined(IRMS_SUPPORT)
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = phase->I_rms[0];
    #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = 0;
    #endif
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 3] = phase->active_power[0];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 4] = phase->active_power[0] >> 16;
    #if defined(REACTIVE_POWER_SUPPORT)
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 5] = phase->reactive_power[0];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 6] = phase->reactive_power[0] >> 16;
    #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 5] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 6] = 0;
    #endif
    #if defined(VA_POWER_SUPPORT)
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 7] = phase->VA_power;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 8] = phase->VA_power >> 16;
    #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 7] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 8] = 0;
    #endif

    #if defined(POWER_FACTOR_SUPPORT)
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 9] = phase->power_factor[0];
    #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 9] = 0;
    #endif
    #if defined(MAINS_FREQUENCY_SUPPORT)
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 10] = phase->frequency;
    #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 10] = 0;
    #endif
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 11] = phase->V_dc_estimate;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 12] = phase->V_dc_estimate >> 16;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 13] = phase->current.I_dc_estimate[0];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 14] = phase->current.I_dc_estimate[0] >> 16;
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 30] = 0;
        prepare_tx_message(31);
        break;
    #if defined(NEUTRAL_MONITOR_SUPPORT)
    case HOST_CMD_GET_READINGS_NEUTRAL:
        /* Exchange voltage, current and power readings (neutral).
           frequency, power factor and reactive power readings. */
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        #if defined(SINGLE_PHASE)
            #if defined(VRMS_SUPPORT)
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = phase->V_rms;
            #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = 0;
            #endif
            #if defined(IRMS_SUPPORT)
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = phase->I_rms[1];
            #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = 0;
            #endif
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 3] = phase->active_power[1];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 4] = phase->active_power[1] >> 16;
            #if REACTIVE_POWER_SUPPORT
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 5] = phase->reactive_power[1];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 6] = phase->reactive_power[1] >> 16;
            #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 5] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 6] = 0;
            #endif
            #ifdef APPARENT_POWER_SUPPORT
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 7] = phase->apparent_power[0];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 8] = phase->apparent_power[0] >> 16;
            #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 7] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 8] = 0;
            #endif
            #if defined(POWER_FACTOR_SUPPORT)
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 9] = phase->power_factor[0];
            #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 9] = 0;
            #endif
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 10] = phase->frequency;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 11] = phase->V_dc_estimate;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 12] = phase->V_dc_estimate >> 16;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 13] = phase->neutral.I_dc_estimate[0];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 14] = phase->neutral.I_dc_estimate[0] >> 16;
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 30] = 0;
        #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = neutral.I_rms;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 3] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 4] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 5] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 6] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 7] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 8] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 9] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 10] = phase->frequency;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 11] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 12] = 0;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 13] = phase->current.I_dc_estimate;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 14] = phase->current.I_dc_estimate >> 16;
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 30] = 0;
        #endif
        prepare_tx_message(31);
        break;
    #endif
    case HOST_CMD_ERASE_FLASH_SEGMENT:
        /* Initialise flash data download, by erasing the area to be used, and setting the
           write pointer. */
        /* There is no checking here to ensure we do not erase inappropriate places. */
        next_flash_loc = (unsigned int *) rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 1];
        flash_clr((int *) next_flash_loc);
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        prepare_tx_message(2);
        break;
    case HOST_CMD_SET_FLASH_POINTER:
        next_flash_loc = (unsigned int *) rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 1];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        prepare_tx_message(2);
        break;
    case HOST_CMD_FLASH_DOWNLOAD:
        if (is_calibration_enabled())
        {
            next_flash_loc = (unsigned int *) rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 1];
            for (i = 2;  i < (rx_len - 12) >> 1;  i++)
                flash_write_int16((int *) next_flash_loc++, rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + i]);
            flash_secure();
            /* Return the next address to the host, so it can check for missed messages */
            tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
            tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
            tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = (unsigned int) next_flash_loc;
            prepare_tx_message(4);
        }
        break;
    case HOST_CMD_FLASH_UPLOAD:
        if (is_calibration_enabled())
        {
            next_flash_loc = (unsigned int *) rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 1];
            tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
            tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
            for (i = 1;  i < MAX_IEC1107_MSG_BODY/2;  i++)
                tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + i] = *next_flash_loc++;
            prepare_tx_message(MAX_IEC1107_MSG_BODY);
        }
        break;
    case HOST_CMD_ZAP_MEMORY_AREA:
        /* Zap memory area (usually this will be flash) */
        if (is_calibration_enabled())
        {
            next_flash_loc = (unsigned int *) rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 1];
            last_flash_loc = (unsigned int *) rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 2];
            while (next_flash_loc < last_flash_loc)
                flash_write_int16((int *) next_flash_loc++, rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 3]);
            flash_secure();
            tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
            tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
            tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = (unsigned int) next_flash_loc;
            prepare_tx_message(4);
        }
        break;
    case HOST_CMD_SUMCHECK_MEMORY:
        /* Sumcheck a specified memory area, and return the sumcheck */
        if (is_calibration_enabled())
        {
            next_flash_loc = (unsigned int *) rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 1];
            last_flash_loc = (unsigned int *) rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 2];
            tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
            tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
            tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = 0;
            while (next_flash_loc < last_flash_loc)
                tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] += *next_flash_loc++;
            flash_secure();
            prepare_tx_message(4);
        }
        break;
    case HOST_CMD_GET_RAW_POWER_PHASE_1:
    #if !defined(SINGLE_PHASE)
    case HOST_CMD_GET_RAW_POWER_PHASE_2:
    case HOST_CMD_GET_RAW_POWER_PHASE_3:
        phase = &chan[rx_msg->uint8[IEC1107_MSG_RX_START_BODY] - 0x91];
    #endif
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = phase->current.P_accum_logged[0][0];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = phase->current.P_accum_logged[0][1];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 3] = phase->current.P_accum_logged[0][2];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 4] = phase->sample_count_logged;
        prepare_tx_message(10);
        break;
    #if defined(REACTIVE_POWER_BY_QUADRATURE_SUPPORT)
    case HOST_CMD_GET_RAW_REACTIVE_POWER_PHASE_1:
        #if !defined(SINGLE_PHASE)
    case HOST_CMD_GET_RAW_REACTIVE_POWER_PHASE_2:
    case HOST_CMD_GET_RAW_REACTIVE_POWER_PHASE_3:
        phase = &chan[rx_msg->uint8[IEC1107_MSG_RX_START_BODY] - 0x95];
        #endif
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = phase->current.P_reactive_accum_logged[0][0];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = phase->current.P_reactive_accum_logged[0][1];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 3] = phase->current.P_reactive_accum_logged[0][2];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 4] = phase->sample_count_logged;
        prepare_tx_message(10);
        break;
    #endif
    #if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
    case HOST_CMD_GET_RAW_POWER_NEUTRAL:
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = phase->neutral.P_accum_logged[0][0];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = phase->neutral.P_accum_logged[0][1];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 3] = phase->neutral.P_accum_logged[0][2];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 4] = phase->sample_count_logged;
        prepare_tx_message(10);
        break;
        #if defined(REACTIVE_POWER_BY_QUADRATURE_SUPPORT)
    case HOST_CMD_GET_RAW_REACTIVE_POWER_NEUTRAL:
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = phase->neutral.P_reactive_accum_logged[0][0];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = phase->neutral.P_reactive_accum_logged[0][1];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 3] = phase->neutral.P_reactive_accum_logged[0][2];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 4] = phase->sample_count_logged;
        prepare_tx_message(10);
        break;
        #endif
    #endif
    case HOST_CMD_CHECK_RTC_ERROR:
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
    #if defined(__MSP430_HAS_TA3__)
        //z = assess_rtc_speed();
    #endif
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = z;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = z >> 16;
        prepare_tx_message(6);
        break;
    #if defined(CORRECTED_RTC_SUPPORT)
    case HOST_CMD_RTC_CORRECTION:
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = rtc_correction;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = rtc_correction >> 16;
        prepare_tx_message(6);
        break;
    #endif
    #if defined(MULTI_RATE_SUPPORT)
    case HOST_CMD_MULTIRATE_SET_PARAMETERS:
        i = multirate_put(&rx_msg->uint8[IEC1107_MSG_RX_START_BODY]);
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = i;
        prepare_tx_message(4);
        break;
    case HOST_CMD_MULTIRATE_GET_PARAMETERS:
        i = multirate_get(&rx_msg->uint8[IEC1107_MSG_RX_START_BODY], &tx_msg.uint8[IEC1107_MSG_TX_START_BODY]);
        prepare_tx_message(i);
        break;
    case HOST_CMD_MULTIRATE_CLEAR_USAGE:
        i = multirate_clear_usage(&rx_msg->uint8[IEC1107_MSG_RX_START_BODY]);
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = i;
        prepare_tx_message(4);
        break;
    case HOST_CMD_MULTIRATE_GET_USAGE:
        i = multirate_get_usage(&rx_msg->uint8[IEC1107_MSG_RX_START_BODY], &tx_msg.uint8[IEC1107_MSG_TX_START_BODY]);
        prepare_tx_message(i);
        break;
    #endif
//////Percy for Calibration 0902////////////////////////////////        
    case HOST_CMD_CALIBRATION_GET_GAIN:
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 1] = rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W + 1]; 
        //ch = rx_msg->uint8[IEC1107_MSG_RX_START_BODY+2];
        //stage = rx_msg->uint8[IEC1107_MSG_RX_START_BODY+3];
    #if !defined(SINGLE_PHASE)    
        phase = &chan[rx_msg->uint8[IEC1107_MSG_RX_START_BODY+2]];
    #else
        if( rx_msg->uint8[IEC1107_MSG_RX_START_BODY+2] == 3 )    //neutral current
        {
          tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = phase_nv->current[1].P_scale_factor[0];
          tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 3] = phase_nv->current[1].Phase_correction[0];          
        }
        else{
    #endif    
    #if GAIN_STAGES > 1
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = phase_nv->current[0].P_scale_factor[rx_msg->uint8[IEC1107_MSG_RX_START_BODY+3]-1];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 3] = phase_nv->current[0].phase_correction[rx_msg->uint8[IEC1107_MSG_RX_START_BODY+3]-1];
    #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 2] = phase_nv->current[0].P_scale_factor[0];
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 3] = phase_nv->current[0].Phase_correction[0];
    #endif
        }
    #ifdef  OFFSET_CORRECTION_SUPPORT
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 4] = phase_nv->current[0].Offset_active_power;               
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 5] = neutral_nv->Offset_active_power;
    #else
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 4] = 0;               
        tx_msg.uint16[IEC1107_MSG_TX_START_BODY_W + 5] = 0;      
    #endif
        prepare_tx_message(12);
        break;                         
    case HOST_CMD_CALIBRATION_SET_GAIN:
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        //ch = rx_msg->uint8[IEC1107_MSG_RX_START_BODY+2];
        //stage = rx_msg->uint8[IEC1107_MSG_RX_START_BODY+3];
        Cal_Gain(rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W+2], rx_msg->uint8[IEC1107_MSG_RX_START_BODY+2], rx_msg->uint8[IEC1107_MSG_RX_START_BODY+3]);
        prepare_tx_message(2);
        break;          
    case HOST_CMD_CALIBRATION_SET_PHASE:
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        //ch = rx_msg->uint8[IEC1107_MSG_RX_START_BODY+2];
        //stage = rx_msg->uint8[IEC1107_MSG_RX_START_BODY+3];
        Cal_Phase(rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W+2], rx_msg->uint8[IEC1107_MSG_RX_START_BODY+2], rx_msg->uint8[IEC1107_MSG_RX_START_BODY+3]);
        prepare_tx_message(2);
        break;          
    case HOST_CMD_CALIBRATION_SET_POFFSET:
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        //ch = rx_msg->uint8[IEC1107_MSG_RX_START_BODY+2];
        //stage = rx_msg->uint8[IEC1107_MSG_RX_START_BODY+3];
        Cal_Poffset(rx_msg->uint16[IEC1107_MSG_RX_START_BODY_W+2], rx_msg->uint8[IEC1107_MSG_RX_START_BODY+2], rx_msg->uint8[IEC1107_MSG_RX_START_BODY+3]);
        prepare_tx_message(2);
        break;           
    case HOST_CMD_CALIBRATION_STORE:      
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY + 1] | 0x80;
        //Cal_DataStore();
        prepare_tx_message(2);
        break;          
        
    default:
        /* For all other message types reply with type 0xFF - bad message type */
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY] = rx_msg->uint8[IEC1107_MSG_RX_START_BODY];
        tx_msg.uint8[IEC1107_MSG_TX_START_BODY + 1] = 0xFF;
        prepare_tx_message(2);
        break;
    }
}
#endif

#if defined(__MSP430__)  &&  (defined(IEC1107_SUPPORT)  ||  defined(SERIAL_CALIBRATION_SUPPORT))  &&  (defined(USART0RX_VECTOR)  ||  defined(USART0RX_ISR))
/* Interrupt to accept IEC1107 messages. */
ISR(USART0RX, serial_rx_interrupt0)
{
    unsigned char ch;
    int i;
    int sum;

    ch = RXBUF0;
    if (char_timeout_1107 == 0)
        rx_msg_ptr = 0;
    char_timeout_1107 = SAMPLES_PER_10_SECONDS/200;
    if (rx_msg_ptr == 0)
    {
        if (ch == 0x68)
        {
            rx_msg.uint8[rx_msg_ptr++] = ch;
            rx_msg_len = 12 + MAX_IEC1107_MSG_BODY;
        }
    }
    else
    {
        if (rx_msg_ptr == 9)
        {
            if (ch <= MAX_IEC1107_MSG_BODY)
                rx_msg_len = 12 + ch;
            else
                rx_msg_ptr = 0;
        }
        rx_msg.uint8[rx_msg_ptr++] = ch;
        if (rx_msg_ptr == rx_msg_len)
        {
            /* End of message */
            sum = rx_msg.uint8[0];
            for (i = 1;  i < rx_msg_len - 2;  i++)
                sum += rx_msg.uint8[i];
            if (rx_msg.uint8[rx_msg_len - 2] == (sum & 0xFF)
                &&
                rx_msg.uint8[rx_msg_len - 1] == 0x16)
            {
                /* Good message received */
                process_rx_message(&rx_msg, rx_msg_len);
            }
            rx_msg_ptr = 0;
        }
    }
}
#endif

/* Interrupt to send IEC1107 messages. */
#if defined(__MSP430__)  &&  (defined(IEC1107_SUPPORT)  ||  defined(SERIAL_CALIBRATION_SUPPORT))  &&  (defined(USART0TX_VECTOR)  ||  defined(USART0TX_ISR))
ISR(USART0TX, serial_tx_interrupt0)
{
    TXBUF0 = tx_msg.uint8[tx_msg_ptr++];
    if (tx_msg_ptr >= tx_msg_len)
    {
        /* Stop transmission */
        U0IE &= ~UTXIE0;
        tx_msg_ptr = 0;
        tx_msg_len = 0;
    }
}
#endif